const CommonUtil = require("../utils/commonUtil");	//NO I18N
const DatabaseUtil = require("../utils/databaseUtil");	//NO I18N
const Scheduler = require("../utils/schedulerUtil");	//NO I18N

const Functions = (function () {
	let reqData;
	let reqParams;
	const _handler = async (data) => {
		_initialize(data);
		let response;
		const handler = reqData.handler.type;
		if (handler === "button_handler") {
			response = await _buttonHandler();
		} else if (handler === "form_handler") {	//NO I18N
			response = await _formHandler();
		}
		return response;
	};

	const _initialize = (data) => {
		reqData = data;
		reqParams = data.params;
	};

	const _buttonHandler = async () => {
		let response = {};
		if (reqData.name === "expensemanagerbtn") {
			switch (reqParams.target.name) {
				case "This Week":	//NO I18N
					response = await _getExpense("Weekly");	//NO I18N
					break;
				case "This Month":	//NO I18N
					response = await _getExpense("Monthly");	//NO I18N
					break;
				case "This Year":	//NO I18N
					response = await _getThisYearExpense();
					break;
				case "Authorize":	//NO I18N
					response = await _handleUserToken();
			}
		}
		return response;
	};

	const _formHandler = async () => {
		try {
			let response = {};
			const formName = reqParams.form.name;
			const formValues = reqParams.form.values;
			const zuid = reqParams.access.user_id;
			if (formName === "addexpenseform") {
				const category = CommonUtil.isEmptyString(formValues.category.label)
					? "Other"	//NO I18N
					: formValues.category.label;

				await DatabaseUtil.expenses.addExpense({
					zuid: zuid,
					date: new Date(formValues.date),
					amount: formValues.amount,
					category: category,
					note: CommonUtil.isEmptyString(formValues.note) ? "" : formValues.note
				});

				const amountSpent = await DatabaseUtil.expenses.getAmountSpentThisWeekAndThisMonth(zuid);
				const currency = await DatabaseUtil.users.getCurrency(zuid);
				let text =
					"Date: " + formValues.date +	//NO I18N
					"\nAmount: " + formValues.amount +	//NO I18N
					" " + currency +
					"\nCategory: " + category;	//NO I18N

				text = CommonUtil.isEmptyString(formValues.note)
					? text
					: text + "\nNote: " + formValues.note;	//NO I18N

				text += "\n\n Amount spent this week: " + amountSpent.thisweek + " " + currency;	//NO I18N
				text += "\n Amount spent this month: " + amountSpent.thismonth + " " + currency;	//NO I18N

				response = {
					text: text,
					card: {
						title: "Expense Added",	//NO I18N
						theme: "modern-inline"	//NO I18N
					}
				};
			} else if (formName === "settingsform") {	//NO I18N
				const zuid = reqParams.access.user_id;
				let update = {};
				let updateDb = false;

				const { currency, notifications } = await DatabaseUtil.users.getUserSettings(zuid);

				if (currency === undefined || currency !== formValues.currency.value) {
					update.currency = formValues.currency.value;
					updateDb = true;
				}

				if (notifications === undefined || notifications.frequency !== formValues.notifications.label) {
					let schedulerId;
					const newFrequency = formValues.notifications.label;
					if (notifications === undefined) {
						schedulerId = await Scheduler.create(zuid, newFrequency);
					} else {
						schedulerId = await Scheduler.update(zuid, newFrequency);
					}
					update.notifications = { frequency: newFrequency, schedulerId };
					updateDb = true;
				}
				let responseText = "No changes made in the settings.";	//NO I18N
				if (updateDb) {
					await DatabaseUtil.users.updateSettings(zuid, update);
					responseText = "Settings updated :)";	//NO I18N
				}

				response = CommonUtil.getBannerResponse(responseText, false);
			}
			return response;
		} catch (error) {
			throw error;
		}
	};

	const _getExpense = async (frequency) => {
		try {
			let fromDate = CommonUtil.getLastMonday();
			let toDate = new Date();
			if (frequency === "Monthly") {
				fromDate = CommonUtil.getFirstDayOfCurrentMonth();
			}

			const allExpenses = await DatabaseUtil.expenses.getExpense(
				reqParams.access.user_id,
				fromDate,
				toDate
			);

			let response;
			if (allExpenses.length === 0) {
				response = CommonUtil.getBannerResponse("No expense found.", true);	//NO I18N
			} else {
				response = CommonUtil.getExpenseResponse(allExpenses, frequency, {
					toDate,
					fromDate,
					month: CommonUtil.getMonthName(fromDate.getMonth() + 1),
					currency: await DatabaseUtil.users.getCurrency(reqParams.access.user_id)
				});
			}
			return response;
		} catch (error) {
			throw error;
		}
	};

	const _getThisYearExpense = async () => {
		try {
			const fromDate = CommonUtil.getFirstDayOfYear();
			const toDate = new Date();
			const overallExpenses = await DatabaseUtil.expenses.getOverallExepenseForYear(
				reqParams.access.user_id,
				fromDate,
				toDate
			);
			
            let rows = [];
			let	totalExpense = 0;
			let response;

			if (overallExpenses.length === 0) {
				response = CommonUtil.getBannerResponse("No expense found.", true);	//NO I18N
			} else {
				overallExpenses.forEach((eachMonth) => {
					totalExpense += eachMonth.totalExpense;
					rows.push({
						Month: CommonUtil.getMonthName(eachMonth._id),
						Amount: eachMonth.totalExpense
					});
				});
			
                response = {
					text: "Total Expense: " + totalExpense,	//NO I18N
					card: {
						title: "Expense for the year " + toDate.getFullYear(),	//NO I18N
						theme: "modern-inline"	//NO I18N
					},
					slides: [
						{
							type: "table",	//NO I18N
							data: {
								headers: ["Month", "Amount"],	//NO I18N
								rows: rows
							}
						}
					]
				};
			}

			return response;
		} catch (error) {
			throw error;
		}
	};

	const _handleUserToken = async () => {
		try {
			const zuid = reqParams.access.user_id;
			await DatabaseUtil.users.addToken(zuid, reqParams.arguments.input.token);

			let notificationFrequency;
			const { currency, notifications } = await DatabaseUtil.users.getUserSettings(zuid);
			if (typeof notifications !== "undefined") {
				notificationFrequency = notifications.frequency === "Weekly" ? "0" : "1";	//NO I18N
			}
			return CommonUtil.getSettingsResponse(currency, notificationFrequency);
		} catch (error) {
			throw error;
		}
	};

	return {
		handler: _handler
	};
})();

module.exports = Functions;
